//! Hangman is a small hangman game that randomly selects a word from the
//! dictionary file and prompts the user to guess the word.

extern crate rand;

use std::error::Error;
use std::fs::File;
use std::path::Path;
use std::io::BufReader;
use std::io::BufRead;
use rand::Rng;
use std::io;

struct HangmanBoard {
    correct_word: String,
    guessed_characters: [bool; 26],
}

impl HangmanBoard {
    fn get_str(&self) -> String {
        let mut board_string = String::new();
        for character in self.correct_word.chars() {
            let char_index = convert_to_ascii_index(character);
            board_string.push(
                match self.guessed_characters[char_index] {
                    true  => character,
                    false => '_',
                }
            );
        }
        return board_string;
    }

    fn is_correct(&self) -> bool {
        for character in self.correct_word.chars() {
            if !self.guessed_characters[convert_to_ascii_index(character)] {
                return false;
            }
        }
        return true;
    }

    fn guess_is_valid(&self, guess: char) -> bool {
        return self.correct_word.contains(guess);
    }
}

fn convert_to_ascii_index(c: char) -> usize {
    return ((c as u8) - ('A' as u8)) as usize;
} 

fn main() {
    let dictionary_path = "dict.txt";

    let guess_word = get_word_to_guess_from_dict(dictionary_path)
        .expect("Error loading specific line from dictionary");
    let mut board = HangmanBoard {correct_word: guess_word, guessed_characters: [false; 26]};

    println!("{}", match play_game(&mut board) {
        true  => "You win!",
        false => "You lose :(",
    });
}

fn play_game(board: &mut HangmanBoard) -> bool {
    let mut trials = 0;
    let max_trials = 7;

    println!("{}", board.get_str());
    while trials < max_trials {
        println!("You have {} attempts remaining", max_trials-trials);
        print_guesses(&board);
        println!("What is your next guess?");
        let mut next_guess = String::new();
        io::stdin().read_line(&mut next_guess)
            .ok().expect("Failed to read input");

        // Convert the guess to a character
        if next_guess.trim().chars().count() > 1 {
            println!("Invalid input");
            continue;
        } else {
            let guess: char  = match next_guess.trim().to_uppercase().chars().next() {
                Some(val) => val,
                None      => panic!("Something is wrong"),
            };
            println!("");

            if !board.guess_is_valid(guess) {
                trials += 1;
            }

            let guess: usize = convert_to_ascii_index(guess);
            board.guessed_characters[guess] = true;
            println!("{}", board.get_str());

            if board.is_correct() {
                return true
            }
        }
   }
   return false;
}

fn print_guesses(board: &HangmanBoard)  {
    let mut guess_list = "You have guessed: ".to_string();
    for i in 0..board.guessed_characters.len() {
        if board.guessed_characters[i] {
            guess_list.push(((i as u8) + ('A' as u8)) as char);
        }
    }
    println!("{}", guess_list);
}

fn get_word_to_guess_from_dict(dict_path: &str) -> Result<String, &str>{
    let dict_path = Path::new(dict_path);

    let dictionary_file = match File::open(&dict_path) {
        Err(why) => panic!("Could not open dictionary file {}: {}",
                           dict_path.display(), Error::description(&why)),
        Ok(file) => file,
    };

    let mut current_line = 1;
    let file_reader = BufReader::new(&dictionary_file);
    let line_number = rand::thread_rng().gen_range(1, 52864);
    for line in file_reader.lines() {
        if current_line != line_number {
            current_line += 1;
            continue;
        }

        return Ok(line.unwrap().to_uppercase());
    }
    return Err("Line number not in file");
}


